(() => {
    function setFormData(formData, el, form) {
        let value;
        let canAppend = false;

        if (el instanceof window.RentecFormElement) {
            if (el.submittable) {
                value = el.submitValue;
                canAppend = true;
            }
        } else {
            // TODO: remove. Form components should be instances of RentecFormElement.
            if (!el.disabled) {
                value = el.value;
                canAppend = true;
            }
        }

        if (canAppend) {
            if (typeof value === 'string') {
                value = value.trim();
            }
            
            try {
                append(formData, el.name, value);
            } catch (e) {
                // Mute these one-off Firefox exceptions that appear to be harmless.
                if (!e.message.includes('Permission denied')) {
                    throw e;
                }
            }
        }

        return formData;
    }
    
    function append(formData, name, value) {
        const isEmpty = value === null || value === undefined || value === '' || (Array.isArray(value) && !value.length);
        
        if (!isEmpty) {
            if (name) {
                if (Array.isArray(value)) {
                    if (value.every(v => v instanceof File)) {
                        value.forEach((v, index) => {
                            // While appending multiple values to the same name is up to spec as a valid way to specify
                            // multiple values for a field, our server does not handle this properly for Files. See docs.
                            formData.append(`${name}[${index}]`, v);
                        });
                    } else {
                        // Multiple values must be appended to the same name. Our server will not always behave as desired
                        // if array indexes are provided. See docs.
                        value.forEach(v => {
                            formData.append(name, v);
                        });
                    }
                } else if (value instanceof File) {
                    formData.append(name, value);
                } else if (value instanceof FormControlValueMap) {
                    value.forEach((key, value) => {
                        append(formData, key, value);
                    });
                } else if (typeof value === 'object' || typeof value === 'function') {
                    throw new Error('Unexpected form value.');
                } else {
                    formData.append(name, value);
                }
            } else {
                if (value instanceof FormControlValueMap) {
                    value.forEach((key, value) => {
                        if (value !== null && value !== undefined) {
                            append(formData, key, value);
                        }
                    });
                }
            }
        } else if (name) {
            // Include empty fields as empty strings in form submissions. This mirrors native form control behavior.
            formData.append(name, '');
        }
    }
    
    class FormControlValueMap {
        constructor(values) {
            this.values = values;
        }
        
        has(name) {
            return Object.prototype.hasOwnProperty.call(this.values, name);
        }
        
        forEach(cb) {
            Object.entries(this.values).forEach(([key, value]) => {
               cb(key, value);
            });
        }
    }

    class FormControlController {
        static getFormData(form) {
            const elements = Array.from(form.querySelectorAll('*'));
            return elements.reduce((map, el) => setFormData(map, el, form), new FormData());
        }

        constructor(host, { reportErrors = true } = {}) {
            this.host = host;
            this.reportErrors = reportErrors;
            host.addController(this);
            
            const form = this.form = host.closest('form, r-form, [form]');

            if (form) {
                this._boundOnFormData = this._onFormData.bind(this);
                form.addEventListener('formdata', this._boundOnFormData);
                
                if (form instanceof HTMLFormElement) {
                    form.reportValidity = () => {
                        const elements = form.querySelectorAll('*');
                        for (const element of elements) {
                            if (typeof element.reportValidity === 'function') {
                                if (!element.reportValidity()) {
                                    return false;
                                }
                            }
                        }
                        return true;
                    }
                    
                    this._boundOnNativeFormSubmit = this._onNativeFormSubmit.bind(this);
                    form.addEventListener('submit', this._boundOnNativeFormSubmit);
                } else {
                    form.register(host);
                    
                    this._boundOnFormError = this._onFormError.bind(this);
                    form.addEventListener('error', this._boundOnFormError);
                    
                    this._boundOnFormSubmit = this._onFormSubmit.bind(this);
                    form.addEventListener('submit', this._boundOnFormSubmit);
                }
            }
        }

        hostDisconnected() {
            if (this.form) {
                this.form.removeEventListener('formdata', this._boundOnFormData);
                
                if (this.form instanceof HTMLFormElement) {
                    this.form.removeEventListener('submit', this._boundOnNativeFormSubmit);
                } else {
                    this.form.removeEventListener('error', this._boundOnFormError);
                    this.form.removeEventListener('submit', this._boundOnFormSubmit);
                    this.form.unregister(this.host);
                }
            }
        }
        
        submitForm() {
            if (this.form) {
                if (this.form instanceof HTMLFormElement) {
                    // TODO: remove the need to support this. All controls should use <r-form>.
                    const submitBtn = this.form.querySelector('button[type="submit"], r-button[type="submit"]');
                    if (submitBtn) {
                        submitBtn.click();
                    } else {
                        // Recreate requestSubmit() due to only somewhat recent browser support.
                        // Note: this doesn't replicate requestSubmit() behavior of triggering the form button, so the above query is still required.
                        const submitEl = document.createElement('input');
                        submitEl.type = 'submit';
                        submitEl.hidden = true;
                        this.form.appendChild(submitEl);
                        submitEl.click();
                        submitEl.remove();
                    }
                } else {
                    if (this.form.submitButton) {
                        this.form.submitButton.click();
                    } else {
                        this.form.submit();
                    }
                }
            }
        }
        
        setFormData(formData) {
            setFormData(formData, this.host, this.form)
        }
        
        _onFormData(e) {
            setFormData(e.formData, this.host, this.form);
        }
        
        _onNativeFormSubmit(e) {
            if (!this.form.reportValidity()) {
                e.preventDefault();
                e.stopImmediatePropagation();
                return false;
            }
        }
        
        _onFormError(e) {
            if (this.reportErrors && this.host.name) {
                this.host.errors = e.detail.errors[this.host.name] || [];
                
                if (this.host.errors.length) {
                    e.detail.consumedErrors.set(this.host.name, this.host);
                }
            }
        }
        
        _onFormSubmit() {
            this.host.errors = [];
        }
    }

    window.FormControlController = FormControlController;
    window.FormControlValueMap = FormControlValueMap;
})();