(() => {
	const { lit } = window;
	const { LitElement, html, css, unsafeCSS } = lit;
	
	const _sheetToStr = sheet => {
		if (sheet) {
			return Array.from(sheet.cssRules).map(rule => rule.cssText).join(' ');
		}
		return '';
	}
	
	const rdsStyles = css`${unsafeCSS(Array.from(document.querySelectorAll('link[ref="rds-stylesheet"]')).reduce((acc, el) => {
		if (el) {
			acc += _sheetToStr(el.sheet);
		}
		return acc;
	}, ''))}`;
	
	window.RentecLitElement = class RentecLitElement extends LitElement {
		static properties = {
			ready: { type: Boolean, state: true },
		}
		static styles = [
			css`
				:host {
					display: block;
				}
			`,
			rdsStyles,
		];
		static options = {
			detectSlots: false,
		}
		
		static define(elementName, elementConstructor = this) {
			if (!window.customElements.get(elementName)) {
				window.customElements.define(elementName, elementConstructor)
			}
		}
		
		static findStyleSheet(sheetName) {
			const stylesheets = Array.from(document.querySelectorAll('head > link'));
			for (const stylesheet of stylesheets) {
				const regExStr = `/(${sheetName})\.css`;
				const regEx = new RegExp(regExStr, 'g');
				if (stylesheet.href && stylesheet.href.match(regEx) && stylesheet.sheet) {
					return css`${unsafeCSS(_sheetToStr(stylesheet.sheet))}`;
				}
			}
			return null;
		}
		
		static isDocumentReady = () => document.readyState === 'interactive' || document.readyState === 'complete';
		static documentReady = new Promise((resolve) => {
			if (RentecLitElement.isDocumentReady()) {
				resolve();
			} else {
				const onReadyStateChangeListener = () => {
					if (RentecLitElement.isDocumentReady()) {
						resolve();
						document.removeEventListener('readystatechange', onReadyStateChangeListener);
					}
				}
				
				document.addEventListener('readystatechange', onReadyStateChangeListener);
			}
		});
		
		willUpdate(changed) {
			if (this._numberProperties.length) {
				this._numberProperties.forEach(([property, definition]) => {
					if (definition.type === Number && typeof this[property] !== 'number') {
						this[property] = parseInt(this[property]) || null;
					}
				});
			}
		}
		
		constructor() {
			super();
			
			if (this.constructor.options.detectSlots) {
				this.slots = new SlotController(this);
			}
			
			this._numberProperties = [];
			this._processProperties();
		}
		
		connectedCallback() {
			super.connectedCallback();
			
			// TODO: remove in favor of isDocumentReady and documentReady, which is a better representation of what this really is.
			this.ready = RentecLitElement.isDocumentReady();
			this.onReady = RentecLitElement.documentReady;
			if (!this.ready) {
				this.onReady.then(() => {
					this.ready = true;
				});
			}
			
			this._validateRequiredProps();
			this.dispatchEvent(new CustomEvent('connected', { bubbles: true, composed: true, detail: { el: this } }));
		}
		
		disconnectedCallback() {
			super.disconnectedCallback();
			this.dispatchEvent(new CustomEvent('disconnected', { bubbles: true, composed: true, detail: { el: this } }));
		}
		
		updated(changed) {
			changed.forEach((oldValue, key) => {
				this._validatePropertyValue(key);
			});
		}
		
		_processProperties() {
			// Automatically initialize properties with default values.
			// A property can also specify its default value in its definition, or explicitly set a value as its default
			// value in its constructor.
			// This prevents one from having to manually initialize all properties, which is not only considered a best
			// practice, but is also a requirement for certain consumers such as Vue which assume elements follow this practice.
			Object.entries(this.constructor.properties).forEach(([property, definition]) => {
				if (Object.prototype.hasOwnProperty.call(definition, 'default')) {
					this[property] = definition.default;
				} else if (definition.type === String) {
					this[property] = null;
				} else if (definition.type === Boolean) {
					this[property] = false;
				} else if (definition.type === Number) {
					this[property] = null;
				} else if (definition.type === Array) {
					this[property] = [];
				} else if (definition.type === Object) {
					this[property] = {};
				} else {
					this[property] = null;
				}
				
				if (definition.type === Number) {
					this._numberProperties.push([property, definition]);
				}
			});
		}
		
		_validatePropertyValue(property) {
			if (!this._validateRequiredProp(this.constructor.properties[property], property)) {
				return;
			}
			
			const value = this[property];
			if (value) {
				const propertyDef = this.constructor.properties[property];
				if (propertyDef.validate || propertyDef.values) {
					let valid = false;
					if (propertyDef.validate) {
						valid = propertyDef.validate.call(this, value);
					}
					if (propertyDef.values) {
						valid = propertyDef.values.includes(value);
					}
					if (!valid || typeof valid === 'string') {
						if (!valid) {
							setTimeout(() => { throw new Error(`Property "${property}" on ${this.nodeName} has an invalid value "${value}"`); });
						} else {
							setTimeout(() => { throw new Error(`Property "${property}" on ${this.nodeName} is invalid: ${valid}`); });
						}
					}
				}
			}
		}
		
		_validateRequiredProps() {
			Object.entries(this.constructor.properties).forEach(([property, definition]) => {
				this._validateRequiredProp(definition, property);
			});
		}
		
		_validateRequiredProp(definition, property) {
			if (definition.required && this[property] == null) {
				setTimeout(() => { throw new Error(`Property "${property}" is required.`); });
				return false;
			}
			return true;
		}
	}
	
	class SlotController {
		get slotted() {
			return Object.keys(this._slots);
		}
		get ready() {
			return RentecLitElement.isDocumentReady();
		}
		
		constructor(host) {
			this.host = host;
			this.handleSlotChange = this.handleSlotChange.bind(this);
			this._slots = {};
			this.onReady = RentecLitElement.documentReady;
			host.addController(this);
		}
		
		hostConnected() {
			this.host.shadowRoot?.addEventListener('slotchange', this.handleSlotChange);
		}
		
		hostDisconnected() {
			this.host.shadowRoot?.removeEventListener('slotchange', this.handleSlotChange);
		}
		
		handleSlotChange(event) {
			const slot = event.target;
			const slotName = slot.name || 'default';
			
			const old = this._slots[slotName];
			this._slots[slotName] = !!slot.assignedNodes().filter(n => n.nodeType === Node.TEXT_NODE ? !!n.textContent.trim().length : true).length;
			
			if (old !== this._slots[slotName]) {
				if (this._slots[slotName]) {
					this.host.setAttribute(`slot-${slotName}`, '');
				} else {
					this.host.removeAttribute(`slot-${slotName}`);
				}
			}
		}
		
		hasSlot(name) {
			return !!this._slots[name];
		}
	}
})();