/**
 * Note: this code should have as much browser support as possible, so that error logging always works.
 */
var Rentec = Rentec || {};

Rentec.logger = new function Logger() {
	var errors = [];
	var canSubmitErrors = false;
	var self = this;
	var disabled = false;
	
	window.onerror = handleErrorEvent;
	window.addEventListener('unhandledrejection', function (event) {
		if (isErrorReportable(event.reason)) {
			self.logError(event.reason);
		}
	});
	window.addEventListener('load', function () {
		// Only send collected errors once the page load event has been fired.
		// Script errors can be thrown under certain cases if a user aborts a page load
		// (e.g. the user reloads the page at just the right time while the page is still loading, or the user
		// cancels the page load at just the right time while the page is still loading).
		// Any loaded scripts may still execute even when the page is aborted, and those scripts may have references
		// to global variables from files that have not yet been fetched/executed.
		canSubmitErrors = true;
		submitJavascriptErrors();
	});
	
	function submitJavascriptErrors() {
		if (!disabled && canSubmitErrors && errors.length > 0) {
			var errorMessages = [];
			for (var i = 0; i < errors.length; i++) {
				var error = errors[i];
				errorMessages.push(
					error.message +
					"\nPage: " + error.page +
					"\nFile: " + (error.file === error.page ? "(inline)" : error.file) +
					(error.lineNumber !== -1 && error.columnNumber !== -1 ? ":" + error.lineNumber + ":" + error.columnNumber : "")
					+ (error.source ? "\n" + error.source : '')
				);
			}
			
			var errorUri = "/";
			var splitUri = window.location.pathname.split("/");
			var depth = splitUri.length - 1;
			
			if (depth > 1 && splitUri[1] !== "vm") {
				errorUri = "/" + splitUri[1] + "/";
			}
			
			j.post(errorUri + "JavaScriptError.action", {
				userAgentString: window.navigator.userAgent,
				errors: errorMessages.join("\n\n")
			});
			
			// Ensure errors are cleared so that future errors don't re-report these previous errors.
			errors = [];
		}
	}
	
	/**
	 * Logs an Error instance to the server.
	 * https://developer.mozilla.org/en-US/docs/web/javascript/reference/global_objects/error
	 * Only name, message, and stack are reliably standard properties of an Error instance.
	 * @param error
	 * @param meta
	 */
	this.logError = function (error, meta) {
		var source, page, columnNumber, lineNumber;
		
		source = error.stack || '(no stack found)';
		
		if (meta && meta.columnNumber) {
			columnNumber = meta.columnNumber;
		} else {
			columnNumber = -1;
		}
		
		if (meta && meta.lineNumber) {
			lineNumber = meta.lineNumber;
		} else {
			lineNumber = -1;
		}
		
		page = location.href.substring(location.href.lastIndexOf("/") + 1, location.href.length);
		
		var errorType = error.name || '(unknown error type)';
		var errorMessage = error.message || '(no error message found)';
		
		errors.push({
			columnNumber: columnNumber,
			file: (meta && meta.file) || '(see stack)',
			lineNumber: lineNumber,
			message: errorType + ": " + errorMessage,
			page: page,
			source: source,
		});
		
		submitJavascriptErrors();
	}
	
	this.disableErrorLogging = function() {
		disabled = true;
	}
	
	/**
	 * Global error event handler. Ensure the handler is set up to receive the 5 arguments from window.onerror.
	 * https://developer.mozilla.org/en-US/docs/Web/API/Window/error_event
	 */
	function handleErrorEvent(message, url, lineNumber, columnNumber, error) {
		// Browser errors triggered by something environmental (not specific to our code) will typically
		// be thrown with an empty or undefined 'url' and a 'lineNumber' of 0. We will capture those and
		// filter based on message contents so that they are not reported to the server:
		if ((!url || url == "") && (!lineNumber || lineNumber == 0)) {
			return false;
		}
		
		// Skip reporting errors originating from files that don't belong to us.
		var location = getLocation(url);
		if (location.origin !== window.location.origin) {
			return false;
		}
		
		if (navigator.userAgent.search("Firefox") != -1) {
			if (message === "Error loading script") {
				// Firefox generates this error when leaving a page before all scripts have finished loading:
				return false;
			}
			
			if (url.includes("aboutSessionRestore.js")) {
				// Firefox 12 seems to have some session restore issues that propagate to restored tabs:
				return false;
			}
		}
		
		if (!isErrorReportable(error)) {
			return false;
		}
		
		self.logError(error, {
			columnNumber: columnNumber,
			file: url,
			lineNumber: lineNumber,
		});
		
		// Prevent the error event from being cancelled by returning false.
		// https://developer.mozilla.org/en-US/docs/Web/API/Window/error_event
		return false;
	}
	
	function getLocation(href) {
		var l = document.createElement("a");
		l.href = href;
		return l;
	}
	
	function isErrorReportable(error) {
		if (!(error instanceof Error)) {
			return false;
		}
		
		// Not all errors come with a stack.
		if (error.stack) {
			// These checks should help to reduce noise.
			if (error.stack.includes('-extension://') || !error.stack.includes(location.origin)) {
				return false;
			}
		}
		
		return true;
	}
}
