(() => {
    const { RentecLitElement, lit } = window;
    const { html, css, nothing } = lit;

    (class RentecButtonComponent extends RentecLitElement {
        static properties = {
            ...super.properties,
            disabled: { type: Boolean },
            loading: { type: Boolean },
            'show-loading-text': { type: Boolean },
            size: { type: String, values: ['xl', 'lg', 'md', 'sm'], default: 'md', reflect: true },
            type: { type: String, values: ['submit'] },
            variant: { type: String, values: ['primary', 'secondary', 'tertiary', 'icon', 'icon-outlined', 'clear', 'link', 'link-muted', 'none'], default: 'none', reflect: true },
            caret: { type: Boolean },
            href: { type: String },
            name: { type: String },
            target: { type: String, values: ['_blank'] },
            tooltip: { type: String },
            confirm: { type: String },
            confirmYes: { type: String, attribute: 'confirm-yes' },
			value: { type: String },
        }

        static styles = [
            super.styles,
            css`
                :host {
                    display: inline-block;
                    width: auto;
                    line-height: 1;
                }
                
                .loading {
                    display: flex;
                    align-items: center;
                    justify-content: center;
                    pointer-events: none;
                }
                
                .loading:not(.loading--text) slot {
                    visibility: hidden;
                }
                
                .btn {
                    display: inline-flex;
                    align-items: center;
                    vertical-align: middle;
                    white-space: nowrap;
                }
                .btn--link, .btn--link-muted {
                    padding: 0;
                    min-height: auto;
                    vertical-align: baseline;
                }
                .btn--link {
                    color: var(--link-color-fg);
                }
                .btn--link-muted {
                    color: var(--color-fg-muted);
                    text-decoration: underline;
                }
                .btn--none {
                    display: inline-block;
                    white-space: initial;
                    padding: 0;
                    min-height: 0;
                    vertical-align: baseline;
                }
            `
        ]

        get btnClasses() {
            const classes = ['btn'];
            if (this.size) {
                classes.push(`btn--${this.size}`);
            }
            if (this.variant) {
                classes.push(`btn--${this.variant}`);
            }
            if (this.disabled) {
                classes.push('btn--disabled');
            }
            return classes.join(' ');
        }

        get loadingStyles() {
            return 'height: 1.5em; width: 1.5em; color: currentColor;';
        }

        get loadingContainerClasses() {
            const classes = [];
            if (!this['show-loading-text']) {
                classes.push('absolute top-0 left-0 w-full h-full flex justify-center items-center');
            } else {
                classes.push('mr-1');
            }
            return classes.join(' ');
        }

        render() {
            const content = html`
                ${this.loading ? html`
                    <span class="${this.loadingContainerClasses}">
                        <r-loading style="${this.loadingStyles}"></r-loading>
                    </span>
                ` : ''}

                <slot name="prefix"></slot>
                <span class="text-center">
                    <slot></slot>
                </span>
                <slot name="suffix">
                    ${this.caret ? html`
                        <r-icon name="down"></r-icon>
                    ` : ''}
                </slot>
            `;

            let button = null;

            if (this.href) {
                button = html`
                    <a
                        class="${this.btnClasses} w-full h-full relative gap-1 ${this.loading ? 'loading' : ''} ${this['show-loading-text'] ? 'loading--text' : ''}"
                        href="${this.href}"
                        target="${this.target ?? nothing}"
                    >
                        ${content}
                    </a>
                `;
            } else {
                button = html`
                    <button
                        id="button"
                        class="${this.btnClasses} w-full h-full relative gap-1 ${this.loading ? 'loading' : ''} ${this['show-loading-text'] ? 'loading--text' : ''}"
                        type="${this.type}" 
                        ?disabled="${this.disabled}" 
                        @click="${this._onClick}"
                    >
                        ${content}
                    </button>
                `
            }

            if (this.tooltip) {
                button = html`
                    <r-tooltip content="${this.tooltip}" delay="${this.variant.includes('icon') ? 500 : null}">
                        ${button}
                    </r-tooltip>
                `;
            }

            return button;
        }

        connectedCallback() {
            super.connectedCallback();
            
            this.form = this.closest('r-form, [form]');
            
            if (this.form) {
                this.form.register(this);
                
                this._onFormSubmitStart = e => {
                    if (e.detail.trigger === this) {
                        this.loading = true;
                    }
                }
                this.form.addEventListener('submitstart', this._onFormSubmitStart);
                this._onFormSubmitEnd = e => {
                    if (e.detail.trigger === this) {
                        this.loading = false;
                    }
                }
                this.form.addEventListener('submitend', this._onFormSubmitEnd);
            }
        }
        disconnectedCallback() {
            this.form?.removeEventListener('submitstart', this._onFormSubmitStart);
            this.form?.removeEventListener('submitend', this._onFormSubmitEnd);
        }
        
        click() {
			if (this.disabled || this.loading) { return; }
			
			if (this.confirm) {
				RentecModalComponent.confirm(this.confirm, this.confirmYes ? { yesText: this.confirmYes } : {}).then(result => {
					if (!result) {
						return;
					}
					
					this._doClick();
				});
			} else {
				this._doClick();
			}
        }

        _onClick(e) {
            e.preventDefault();
            e.stopImmediatePropagation();
			
			this.click();
        }
        
        _doClick() {
			this.dispatchEvent(new CustomEvent('click', {
				bubbles: true
			}));
			
			if (this.type === 'submit' && this.form) {
				const options = {
					trigger: this,
				}
				
				if (this.name) {
					options.appendData = {
						[this.name]: this.value ? this.value : true,
					}
				}
				
				this.form.submit(options);
			}
        }
    }).define('r-button');
})();
