(() => {
    const { lit, RentecFormElement } = window;
    const { html, nothing, css } = lit;
    
    (class RentecInputComponent extends RentecFormElement {
        static properties = {
            ...super.properties,
            size: { type: String, values: ['md', 'sm'], default: 'md', reflect: true },
            type: { type: String, values: ['email', 'password', 'text'], default: 'text', reflect: true },
            light: { type: Boolean },
            autocomplete: { type: String, default: 'off', },
            autofocus: { type: Boolean },
            maxlength: { type: Number },
            clearable: { type: Boolean },
            _wrapped: { type: Boolean },
            _popup: { type: Boolean },
            _showpopup: { type: Boolean },
            _nativetype: { type: String },
            _isShowPassword: { type: Boolean, state: true },
            _dateValue: { type: String, state: true },
            focused: { type: Boolean, state: true },
            floatingLabel: { type: Boolean, attribute: 'floating-label' },
        };
        
        static styles = [
            super.styles,
            css`
                :host {
                    --input-wrapped-padding-y-sm: calc((var(--control-height-sm) - var(--space-6) - (var(--border-width-default) * 2)) / 2);;
                    --input-wrapped-padding-y-md: calc((var(--control-height-md) - var(--space-8) - (var(--border-width-default) * 2)) / 2);;
                }
                
                :host(:not([slot-label]):not([label])) .label {
                    display: none;
                }
                
                :host(:not([slot-help]):not([help])) .help {
                    display: none;
                }
                
                .prefix {
                    display: flex;
                    gap: var(--space-1);
                    padding: 0 var(--space-2);
                }
                :host(:not([slot-prefix])) .prefix {
                    display: none;
                }
                
                .suffix {
                    display: flex;
                    padding: 0 var(--space-2);
                    align-items: center;
                    gap: var(--space-1);
                }
                :host(:not([slot-suffix]):not([type="password"])) .suffix {
                    display: none;
                }
                
                :host([slot-prefix]) .input {
                    padding-left: 0;
                }
                :host([slot-suffix]) .input, :host([type="password"]) .input {
                    padding-right: 0;
                }
                
                .input-wrapper {
                    background-color: var(--color-bg-default);
                    border: var(--border-width-default) solid var(--control-color-border);
                    border-radius: var(--border-radius-md);
                    
                    transition: 250ms ease;
                    transition-property: color, background-color, box-shadow, border-color;
                    
                    width: 100%;
                    max-width: 100%;
                    
                    display: flex;
                    align-items: center;
                }
                .input-wrapper--disabled {
                    color: var(--control-color-fg-disabled);
                    background-color: var(--control-color-bg-disabled);
                    cursor: not-allowed;
                }
                .input-wrapper--focused {
                    border-color: var(--color-primary-light);
                    box-shadow: 0 0 0 1px var(--color-primary-light);
                    outline: none;
                }
                .input-wrapper--error {
                    border-color: var(--color-fg-danger);
                }
                
                .wrapped {
                    padding: var(--input-wrapped-padding-y-sm) 0;
                    display: flex;
                    align-items: center;
                    flex-wrap: wrap;
                    column-gap: var(--space-1);
                    row-gap: var(--space-1);
                    width: 100%;
                }
                .wrapped .input {
                    padding: calc(var(--control-padding-y-sm) - var(--input-wrapped-padding-y-sm)) var(--control-padding-x-sm);
                    min-width: var(--space-16);
                    width: auto;
                    flex-grow: 1;
                }
                :host([size="md"]) .wrapped {
                    padding: var(--input-wrapped-padding-y-md) 0;
                }
                :host([size="md"]) .wrapped .input {
                    padding: calc(var(--control-padding-y-md) - var(--input-wrapped-padding-y-md)) var(--control-padding-x-md);
                }
                :host([slot-input-wrap]) .wrapped {
                    padding-left: var(--space-1);
                }
                
                .input, .input:focus {
                    border: none;
                    outline: none;
                    box-shadow: none;
                }
                :host([slot-input-wrap]) .input {
                    padding-left: var(--space-1);
                }
                
                ::slotted(input) {
                    outline: none !important;
                    box-shadow: none !important;
                    border: none !important;
                }
                
                .append {
                    background-color: var(--color-bg-muted);
                    height: 100%;
                    display: flex;
                    align-items: center;
                    padding: 0 var(--space-2);
                    border-top-right-radius: var(--border-radius-md);
                    border-bottom-right-radius: var(--border-radius-md);
                }
                :host(:not([slot-append])) .append {
                    display: none;
                }
                
                .Label--floating {
                    position: absolute;
                    z-index: 1;
                    font-size: 14px;
                    line-height: 1;
                    left: 16px;
                    top: -8px;
                    background: var(--control-color-bg);
                    padding: 0 var(--space-1);
                    border-radius: var(--border-radius-md);
                }
            `
        ];

        render() {
            const inputWrapper = html`
                <div class="flex relative" style="align-items: stretch;">
                    <div class="input-wrapper ${this.focused ? 'input-wrapper--focused' : ''} ${this.disabled ? 'input-wrapper--disabled' : ''} ${this.errors.length ? 'input-wrapper--error' : ''}" part="input-wrapper ${this.focused ? 'input-wrapper--focused' : ''}" @click="${this.focus}">
                        <div class="prefix">
                            <slot name="prefix"></slot>
                        </div>
                        
                        <div class="${this._wrapped ? 'wrapped' : ''}" style="width: 100%; min-width: 0;">
                            <slot name="input-wrap"></slot>
                            <slot name="input" @slotchange="${this._onInputSlotChange}">
                                <input
                                    id="${this._id}"
                                    class="input ${this.size ? `input--${this.size}` : ''}"
                                    part="input"
                                    name="${this.name ?? nothing}"
                                    .value="${this._type === 'date' || this._type === 'datetime-local' ? (this._dateValue || '') : this.value}"
                                    ?disabled="${this.disabled}"
                                    ?required="${this.required}"
                                    maxlength="${this.maxlength ?? nothing}"
                                    type="${this._type}"
                                    autocomplete="${this.autocomplete}"
                                    placeholder="${this.placeholder ?? nothing}"
                                    autocapitalize="${this.autocapitalize}"
                                    spellcheck="${this.spellcheck}"
                                    @input="${this._onInput}"
                                    @focus="${this._onFocus}"
                                    @blur="${this._onBlur}"
                                >
                            </slot>
                        </div>
                        
                        <div class="suffix">
                            ${this._showClear ? html`
                                <style>
                                    .suffix {
                                        display: inherit !important;
                                    }
                                </style>
                                <r-button variant="icon" @click="${this._onClearClick}">
                                    <r-icon name="x"></r-icon>
                                </r-button>
                            ` : ''}
                            <slot name="suffix">
                                ${this.type === 'password' ? html`
                                    <r-button variant="icon" @click="${this._onPasswordToggleClick}">
                                        ${this._isShowPassword ? html`
                                            <r-icon name="eyex"></r-icon>
                                        ` : html`
                                            <r-icon name="eye"></r-icon>
                                        `}
                                    </r-button>
                                ` : ''}
                            </slot>
                        </div>
                        
                        <div class="append">
                            <slot name="append"></slot>
                        </div>
                    </div>
                </div>
            `;
            
            return html`
                <div class="label relative leading-none ${!this.floatingLabel ? 'mb-1' : ''}">
                    <label for="${this._id}" class="Label Label--${this.size} ${this.floatingLabel ? 'Label--floating' : ''}">
                        <slot name="label">
                            ${this.label}
                        </slot>
                        ${this.required && !this.hideRequired ? html`<span style=" color: var(--color-fg-danger);">*</span>` : ''}
                    </label>
                </div>
                
                ${this._popup ? html`
                    <r-popup ?show="${this._showpopup}">
                        ${inputWrapper}
                        <div slot="content">
                            <slot name="popup-content"></slot>
                        </div>
                    </r-popup>
                ` : html`
                    ${inputWrapper}
                `}

                <div class="help color-fg-muted f5 mt-1">
                    <slot name="help">
                        <div>${this.help}</div>
                    </slot>
                </div>

                ${this.errors?.map(e => html`
                    <r-form-error exportparts="container:error-container, icon:error-icon">${e}</r-form-error>
                `)}
            `;
        }

        get _inputEl() { return this._slottedInput || this.shadowRoot?.querySelector(`#${this._id}`); }

        get autocapitalize() {
            return 'none';
        }
        get spellcheck() {
            return 'false';
        }
        
        get _id() {
            return 'input';
        }
        
        get _type() {
            if (this.type === 'password') {
                if (this._isShowPassword) {
                    return 'text';
                }
                return this.type;
            }
            if (this.type === 'email') {
                return this.type;
            }
            
            if (this._nativetype) {
                return this._nativetype;
            }
            
            return 'text';
        }
        
        get _showClear() {
            return this.clearable && !!this.value;
        }
        
        firstUpdated() {
            this.dispatchEvent(new CustomEvent('ready'));
        }
        
        constructor() {
            super();
            
            window.addEventListener('load', () => {
                if (this.autofocus) {
                    this.updateComplete.then(() => {
                        this.focus();
                    });
                }
            });
        }
        
        connectedCallback() {
            super.connectedCallback();
            
            this._onKeyDown = (e) => {
                if (e.key === 'Enter') {
                    this.dispatchEvent(new CustomEvent('input-enter'));
                    this.triggerSubmit();
                }
            };
            this.addEventListener('keydown', this._onKeyDown);
        }
        
        disconnectedCallback() {
            super.disconnectedCallback();
            this.removeEventListener('keydown', this._onKeyDown);
        }
        
        updated(changed) {
            super.updated(changed);
            
            if (this._slottedInput) {
                if (!this._registeredSlottedInput) {
                    this._registeredSlottedInput = true;
                    this._slottedInput.addEventListener('focus', () => this._onFocus());
                    this._slottedInput.addEventListener('blur', () => this._onBlur());
                    this._slottedInput.addEventListener('input', (e) => {
                        this.value = e.target.value;
                    });
                }
                
                this._slottedInput.type = this._type;
                this._slottedInput.value = this.value;
                this._slottedInput.autocomplete = this.autocomplete;
                this._slottedInput.autocapitalize = this.autocapitalize;
                this._slottedInput.required = this.required;
                this._slottedInput.placeholder = this.placeholder || '';
                
                // Setting a native <input>'s spellcheck property to anything will set the property to true.
                this._slottedInput.setAttribute('spellcheck', this.spellcheck);
            }
            
            if (changed.has('light') && this.light && !this._injectedInput) {
                const input = document.createElement('input');
                this._inputEl.classList.forEach(cls => cls && input.classList.add(cls));
                input.setAttribute('slot', 'input');
                this.append(input);
                
                if (this.label) {
                    const id = `input_${this.label.toLowerCase()}`;
                    
                    const label = document.createElement('label');
                    label.classList.add('Label');
                    label.setAttribute('for', id);
                    label.textContent = this.label;
                    label.setAttribute('slot', 'label');
                    this.append(label);
                    
                    input.id = id;
                }
                
                this._injectedInput = true;
            }
        }

        checkValidity() {
            if (this.type === 'email') {
                return this._checkRequiredValuePresent() && this._inputEl.checkValidity();
            }
            return this._checkRequiredValuePresent();
        }
        reportValidity() {
            return this._reportValidity(() => {
                this._reportRequiredValuePresent();
                
                if (this.type === 'email' && !this.errors.length && !this._inputEl.checkValidity()) {
                    this.errors = ['Invalid e-mail'];
                }
            });
        }

        focus() {
            this._inputEl.focus();
        }
        
        clear() {
            this.value = '';
            this._inputEl.value = '';
            this.dispatchEvent(new CustomEvent('input'));
        }

        _onInput() {
            this.value = this._inputEl.value;
        }
        _onFocus() {
            this.focused = true;
        }
        _onBlur() {
            this.focused = false;
            this.dispatchEvent(new CustomEvent('blur'));
        }
        
        _onPasswordToggleClick() {
            this._isShowPassword = !this._isShowPassword;
        }
        
        _onInputSlotChange(e) {
            this._slottedInput = e.target.assignedElements()[0];
        }
        
        _onClearClick() {
            this.clear();
        }
    }).define('r-input');
})();
