(() => {
    const { RentecLitElement, lit, overlayManager, fetcher, rodash } = window;
    const { html, css, render, unsafeHTML, ref, createRef } = lit;

    const MODAL_FADE_DURATION_MS = 100;
    const MODAL_SLIDE_DURATION_MS = 150;
    
    class RentecModalComponent extends FormMixin(RentecLitElement) {
        modalRef = createRef();
        modalBackdropRef = createRef();
        modalPanelContainerRef = createRef();
        
        static properties = {
            ...super.properties,
            top: { type: Boolean },
            inescapable: { type: Boolean },
            show: { type: Boolean },
            heading: { type: String },
            source: {
                type: String,
                hasChanged: () => {
                    // We want to load the source whenever the source is updated, even if it's the same source, as there
                    // are cases where this is desired.
                    return true;
                }
            },
            errors: { type: Array },
            maxWidth: { type: String, values: ['sm', 'md', 'lg'], attribute: 'max-width', reflect: true },
            minWidth: { type: String, values: ['sm', 'md', 'lg'], attribute: 'min-width', reflect: true },
            width: { type: String, values: ['sm', 'md', 'lg', 'full'], reflect: true, validate: function(value) {
                if (value && (this.minWidth || this.maxWidth)) {
                    setTimeout(() => {
                       throw new Error('Warning: setting a width does not make sense when there is a min/max width set.')
                    });
                }
            } },
            height: { type: String, values: ['full'], reflect: true },
            form: { type: String, reflect: true },
            staticForm: { type: Boolean, attribute: 'static-form' },
            layout: { type: String, values: ['center', 'top', 'fullscreen'], default: 'center', reflect: true },
            sizing: { type: String, values: ['md', 'sm'], default: 'md', reflect: true },

            _isLoadingSource: { type: Boolean, state: true },
            _isLoadingSourceError: { type: Boolean, state: true },
        }
        
        static styles = [
            super.styles,
            css`
                :host {
                    --modal-panel-padding: var(--space-6);
                    --modal-panel-padding-sm: var(--space-4);
                    --modal-panel-width-sm: calc(var(--space-s-step) * 2);
                    --modal-panel-width-md: calc(var(--space-s-step) * 5);
                    --modal-panel-width-lg: calc(var(--space-s-step) * 9);
                    
                    /* Remove from document flow */
                    position: absolute;
                }
                
                :host([min-width="sm"]) .Modal_panel {
                    min-width: var(--modal-panel-width-sm);
                }
                :host([min-width="md"]) .Modal_panel {
                    min-width: var(--modal-panel-width-md);
                }
                :host([min-width="lg"]) .Modal_panel {
                    min-width: var(--modal-panel-width-lg);
                }
                :host([max-width="sm"]) .Modal_panel {
                    max-width: var(--modal-panel-width-sm);
                }
                :host([max-width="md"]) .Modal_panel {
                    max-width: var(--modal-panel-width-md);
                }
                :host([max-width="lg"]) .Modal_panel {
                    max-width: var(--modal-panel-width-lg);
                }
                :host([width="sm"]) .Modal_panel {
                    max-width: var(--modal-panel-width-sm);
                    width: 100%;
                }
                :host([width="md"]) .Modal_panel {
                    max-width: var(--modal-panel-width-md);
                    width: 100%;
                }
                :host([width="lg"]) .Modal_panel {
                    max-width: var(--modal-panel-width-lg);
                    width: 100%;
                }
                :host([width="full"]) .Modal_panel {
                    width: 100%;
                }
                :host([height="full"]) .Modal_panelContainer {
                    align-items: revert;
                    height: 100vh;
                }
                :host([height="full"]) .Modal_panelContent {
                    height: 100%;
                    overflow: auto;
                }
                :host([top]:not([height="full"],[layout="fullscreen"])) .Modal_panelContainer, :host([layout="top"]:not([height="full"],[layout="fullscreen"])) .Modal_panelContainer {
                    align-items: start;
                }
                :host([layout="fullscreen"]) .Modal_panelContainer {
                    align-items: revert;
                    height: 100vh;
                    padding: 0;
                }
                :host([layout="fullscreen"]) .Modal_panelContent {
                    height: 100%;
                    overflow: auto;
                }
                :host([layout="fullscreen"]) .Modal_panel {
                    margin: 0;
                    width: 100%;
                    border-radius: 0;
                }
                :host([sizing="sm"]) {
                    --modal-panel-padding: var(--modal-panel-padding-sm);
                }
                :host([sizing="sm"]) .Modal_panelHeading {
                    font-size: var(--fs-f4);
                    line-height: var(--fs-f4-lh);
                }
                :host([sizing="sm"]) .Modal_panelFooter {
                    margin-top: var(--space-4);
                    padding: var(--space-4) var(--modal-panel-padding);
                }
                
                .initial {
                    all: initial;
                }
                
                .Modal {
                    position: fixed;
                    inset: 0;
                    z-index: var(--z-dialog);
                    display: flex;
                    align-items: start;
                    justify-content: center;
                    overscroll-behavior: contain;
                    overflow: auto;
                    font-family: var(--ff-primary);
                    user-select: text;
					color: var(--color-fg-default);
                }
                
                .Modal_backdrop {
                    position: fixed;
                    inset: 0;
                    background-color: var(--color-bg-backdrop);
                }
                
                .Modal_panelContainer {
                    display: flex;
                    align-items: center;
                    justify-content: center;
                    padding: var(--space-2);
                    min-height: 100%;
                    text-align: center;
                    width: 100%;
                }
                
                .Modal_panel {
                    position: relative;
                    text-align: left;
                    margin-top: var(--space-8);
                    margin-bottom: var(--space-8);
                    background-color: var(--color-bg-default);
                    padding: var(--modal-panel-padding);
                    border-radius: var(--border-radius-md);
                    border: var(--modal-border, none);
                    display: flex;
                    flex-direction: column;
                    min-width: 0; /* prevent panel from blowing out of panelContainer */
                }
                
                .Modal_panelHeader {
                    display: flex;
                    align-items: start;
                    gap: var(--modal-panel-padding);
                    margin-bottom: var(--space-2);
                }
                
                .Modal_panelHeading {
                    flex: auto;
                    font-size: var(--fs-h4);
                    line-height: var(--fs-f4-lh);
                    font-weight: bold;
                    /* Account for modal width being smaller than heading. */
                    min-width: 0;
                    word-break: break-word;
                }
                
                .Modal_panelContent {
                    flex: auto;
                }
                
                .Modal_panelFooter {
                    background-color: var(--color-bg-muted);
                    margin: calc(var(--modal-panel-padding) * -1);
                    margin-top: var(--space-8);
                    padding: var(--space-4) var(--modal-panel-padding);
                    display: flex;
                    justify-content: end;
                    gap: var(--space-2);
                    border-bottom-left-radius: var(--border-radius-md);
                    border-bottom-right-radius: var(--border-radius-md);
                }
                
                :host(:not([slot-buttons])) .footer {
                    display: none;
                }
                
                :host(:not([sizing="sm"])) slot[name="buttons"]::slotted(r-button), :host(:not([sizing="sm"])) slot[name="buttons-aux"]::slotted(r-button) {
                    --btn-md-padding: var(--btn-lg-padding);
                    --btn-md-height: var(--btn-lg-height);
                }
            `
        ]
        
        static options = {
            detectSlots: true,
        }

        render() {
            return html`
                <div ${ref(this.modalRef)} class="Modal initial" role="dialog" aria-modal="true" style="display: ${this.show ? '' : 'none'};">
                    <div ${ref(this.modalBackdropRef)} class="Modal_backdrop" @click="${this._onClickOutside}"></div>
                    <div ${ref(this.modalPanelContainerRef)} class="Modal_panelContainer">
                        <div class="Modal_panel">
                            ${!this.inescapable || this.heading ? html`
                                <div class="Modal_panelHeader">
                                    <div class="Modal_panelHeading">${this.heading}</div>
                                    <div class="flex gap-1">
                                        <slot name="header-actions"></slot>
                                        ${!this.inescapable ? html`
                                            <r-button variant="icon" size="${this.sizing}" @click="${this.close}">
                                                <r-icon name="x"></r-icon>
                                            </r-button>
                                        ` : ''}
                                    </div>
                                </div>
                            ` : ''}
                            <div class="Modal_panelContent">
                                ${this.errors.length ? html`
                                    <div id="errors" class="flex flex-col gap-1 w-full mb-4">
                                        ${this.errors?.map(e => html`
                                            <r-notice class="f4" variant="danger">${e}</r-notice>
                                        `)}
                                    </div>
                                ` : ''}
                                
                                ${this.source ? html`
                                    ${this._isLoadingSource ? html`
                                        <div class="flex justify-center">
                                            <r-loading class="w-24 h-24"></r-loading>
                                        </div>
                                    ` : html`
                                        <slot></slot>
                                    `}
                                    ${this._isLoadingSourceError ? html`
                                        <p>Failed to load content.</p>
                                    ` : ''}
                                ` : html`
                                    <slot></slot>
                                `}
                            </div>
                            
                            ${!this._isLoadingSource && !this._isLoadingSourceError ? html`
                                <div class="footer Modal_panelFooter">
                                    <div class="flex w-full gap-btns">
                                        <div class="flex-auto flex gap-btns">
                                            <slot name="buttons-aux" @click="${this._onButtonsClick}"></slot>
                                        </div>
                                        <div class="flex gap-btns">
                                            <slot name="buttons" @click="${this._onButtonsClick}"></slot>
                                        </div>
                                    </div>
                                </div>
                            ` : ''}
                        </div>
                    </div>
                </div>
                
                ${this.form ? html`${this._formEl}` : ''}
            `;
        }

        connectedCallback() {
            super.connectedCallback();
            
            this._addFormEventListeners();
            
            this._boundHandleModalOpenEvent = this._handleModalOpenEvent.bind(this)
            window.addEventListener('modal-open', this._boundHandleModalOpenEvent);
        }

        disconnectedCallback() {
            super.disconnectedCallback();
            
            window.removeEventListener('modal-open', this._boundHandleModalOpenEvent);
            
            // Modal close logic needs to run.
            this.show = false;
        }

        updated(changed) {
            super.updated(changed);
            
            if (changed.has('show')) {
                if (!changed.get('show') && this.show) {
                    // Scroll is not reset when modal is shown.
                    this.modalRef.value.scrollTop = 0;
                    this._openModal();
                } else if (changed.get('show') && !this.show) {
                    this._closeModal();
                }
            }
            
            if (changed.has('staticForm')) {
                this._setStatic(this.staticForm);
            }
            if (changed.has('form')) {
                this._setAction(this.form);
            }
            
            if (changed.has('source') && !changed.has('show') && this.source && this.show) {
                // Reload the source when the Modal is already showing.
                this._loadSource();
            }
        }

        open() {
            this.show = true;
        }
        close() {
            this.show = false;
        }

        _openModal() {
            if (this.source) {
                this._loadSource();
            }
            
            overlayManager.add(this, () => {
                if (!this.inescapable) {
                    this.close();
                }
            });

            this.dispatchEvent(new CustomEvent('open'));
            
            this.modalBackdropRef.value.animate([
                { opacity: 0 },
                { opacity: 100 },
            ], { duration: MODAL_FADE_DURATION_MS, easing: 'linear' });
            
            if (this.layout !== 'fullscreen') {
                this.modalPanelContainerRef.value.animate([
                    { opacity: 0 },
                    { opacity: 100 },
                ], { duration: MODAL_FADE_DURATION_MS, easing: 'linear' });
            } else {
                this.modalPanelContainerRef.value.animate([
                    { transform: 'translateY(100%)' },
                    { transform: 'translateY(0)' },
                ], { duration: MODAL_SLIDE_DURATION_MS, easing: 'linear' });
            }
        }

        _closeModal() {
            overlayManager.remove(this);
            this.dispatchEvent(new CustomEvent('close'));
        }

        _handleModalOpenEvent(e) {
            if (e.detail.modal === this.id) {
                if (e.detail.source) {
                    this.source = e.detail.source;
                }
                if (e.detail.heading) {
                    this.heading = e.detail.heading;
                }
                
                this.open();
            }
        }

        _onButtonsClick(e) {
            if (e.target.dataset.modalClose) {
                this.close();
            }
        }

        _onClickOutside() {
            if (!this.inescapable) {
                this.close();
            }
        }

        async _loadSource() {
            this._isLoadingSourceError = false;
            this._isLoadingSource = true;

            // Clear out any previously fetched content.
            Array.from(this.childNodes).forEach(el => {
                if (el._sourceFetched) {
                    el.remove();
                }
            });
            
            if (this._loadSourceAbortController) {
                this._loadSourceAbortController.abort();
            }
            
            this._loadSourceAbortController = new AbortController();
            
            try {
                const res = await fetcher.fetch(this.source, { controller: this._loadSourceAbortController });
                
                if (res.canceled) {
                    return;
                }
                
                htmlToEls(res.data).forEach(el => {
                    el._sourceFetched = true;
                    this.append(el);
                });
                
                // Prevent issues where dynamic scripts fire before the elements have actually been rendered into the DOM.
                // This is a rare race condition related to dynamic script injection, so it is generally advised to wait a tick before executing such scripts for robustness, in the event that those
                // dynamic scripts rely on the presence of fetched elements.
                setTimeout(() => {
                    this._handleSourceScripts();
                });
                
                this._loadSourceAbortController = null;
                this._isLoadingSource = false;
            } catch (e) {
                this._isLoadingSourceError = true;
                this._isLoadingSource = false;
            }
        }
        
        /**
         * Simply appending script elements to the DOM won't execute them. Need to do some special logic to execute them.
         */
        _handleSourceScripts() {
            const externalScripts = [];
            const inlineScripts = [];
            
            Array.from(this.querySelectorAll('script'))
                .forEach(oldScriptEl => {
                    const newScriptEl = document.createElement('script');
                    newScriptEl.type = oldScriptEl.type;
                    newScriptEl._sourceFetched = true;
                    newScriptEl.appendChild(document.createTextNode(oldScriptEl.innerHTML));
                    
                    if (oldScriptEl.src) {
                        newScriptEl.src = oldScriptEl.src;
                        
                        // Ensure external scripts are loaded in order.
                        newScriptEl.async = false;
                        
                        externalScripts.push(new Promise((resolve) => {
                            newScriptEl.addEventListener('load', () => {
                                resolve();
                            })
                        }));
                        oldScriptEl.parentNode.replaceChild(newScriptEl, oldScriptEl);
                    } else {
                        inlineScripts.push({ oldScriptEl, newScriptEl });
                    }
                });
            
            // Defer execution of inline scripts until external scripts have loaded.
            if (externalScripts.length) {
                Promise.all(externalScripts).then(() => {
                    inlineScripts.forEach(({ oldScriptEl, newScriptEl }) => {
                        if (!oldScriptEl.parentNode) {
                            // Ensure the script element is still on the DOM.
                            // It may not be if the script element was removed (e.g., if a new source was loaded for the modal).
                            return;
                        }
                        
                        oldScriptEl.parentNode.replaceChild(newScriptEl, oldScriptEl);
                    });
                });
            } else {
                inlineScripts.forEach(({ oldScriptEl, newScriptEl }) => {
                    oldScriptEl.parentNode.replaceChild(newScriptEl, oldScriptEl);
                });
            }
        }
        
        _onFormSubmit() {
            this.errors = [];
        }
        
        async _onFormError(e) {
            this.errors = e.detail.errors;
            
            e.detail.reported();
            
            const errors = this.shadowRoot.querySelector('#errors');
            if (errors) {
                if (!(await rodash.isElementFullyInView(errors))) {
                    errors.scrollIntoView({ behavior: 'smooth' });
                }
            }
        }

        static confirm(message, { yesText = 'OK' } = {}) {
            return new Promise((resolve) => {
                const ok = () => {
                    // Resolve first, otherwise the modal will close and trigger the cancel handler first before this can resolve.
                    resolve(true);
                    render(html``, document.body);
                }

                const cancel = () => {
                    resolve(false);
                    render(html``, document.body);
                }

                const modal = html`
                    <r-modal min-width="sm" max-width="md" dismissable show @close="${cancel}">
                        <div>${html`${unsafeHTML(message.replaceAll('\\n', '<br>'))}`}</div>
                        <r-button slot="buttons" variant="tertiary" size="lg" @click="${cancel}">Cancel</r-button>
                        <r-button slot="buttons" variant="primary" size="lg" @click="${ok}">${yesText}</r-button>
                    </r-modal>
                `;

                render(modal, document.body);
            });
        }
    }

    RentecModalComponent.define('r-modal');
    window.RentecModalComponent = RentecModalComponent;
    
    function FormMixin(superclass) {
        return class extends superclass {
            _formEl = document.createElement('r-form');
            
            get submitButton() {
                return this._formEl.submitButton;
            }
            
            register(element) {
                this._formEl.register(element);
            }
            unregister(element) {
                this._formEl.unregister(element);
            }
            
            submit(options) {
                this._formEl.submit(options);
            }
            
            _setAction(val) {
                this._formEl.action = val;
            }
            
            _setStatic(val) {
                this._formEl.static = val;
            }
            
            _addFormEventListeners() {
                this._formEl.addEventListener('formdata', e => {
                    const event = new CustomEvent('formdata');
                    event.formData = e.formData;
                    this.dispatchEvent(event);
                });
                this._formEl.addEventListener('submit', this._onFormSubmit.bind(this));
                this._formEl.addEventListener('error', e => this.dispatchEvent(new CustomEvent('error', { detail: e.detail })));
                this._formEl.addEventListener('general-error', this._onFormError.bind(this));
                this._formEl.addEventListener('submitstart', e => this.dispatchEvent(new CustomEvent('submitstart', { detail: e.detail })));
                this._formEl.addEventListener('submitend', e => this.dispatchEvent(new CustomEvent('submitend', { detail: e.detail })));
				this._formEl.addEventListener('submitted', e => this.dispatchEvent(new CustomEvent('submitresponse', { detail: e.detail })));
            }
        };
    }

    (class RentecModalTriggerComponent extends RentecLitElement {
        static properties = {
            ...super.properties,
            modal: { type: String },
            ['modal-source']: { type: String },
            ['modal-heading']: { type: String },
        }
        
        static styles = [
            super.styles,
            css`
                :host {
                    display: contents;
                }
            `,
        ];

        render() {
            return html`<slot @click="${this._onClick}"></slot>`;
        }
        
        trigger() {
            window.dispatchEvent(new CustomEvent('modal-open', {
                detail: {
                    modal: this.modal,
                    source: this['modal-source'],
                    heading: this['modal-heading'],
                },
            }))
        }
        
        _onClick(e) {
            e.preventDefault();
            e.stopImmediatePropagation();
            
            this.trigger();
        }
    }).define('r-modal-trigger');

    function htmlToEls(html) {
        const template = document.createElement('template');
        html = html.trim(); // Never return a text node of whitespace as the result
        template.innerHTML = html;
        return Array.from(template.content.childNodes);
    }
})();
