(() => {
    const { RentecLitElement, lit, FloatingUIDOM } = window;
    const { html, css } = lit;
    
    const POPUP_OFFSET = 8;
    const POPUP_TRANSITION_MS = 25;
	const POPUP_CLOSE_DELAY_MS = 100;

    (class RentecPopup extends RentecLitElement {
		_slottedTriggerEl = null;
		
        static properties = {
            ...super.properties,
			placement: { type: String, values: ['top', 'right', 'bottom', 'left', 'bottom-start'], default: 'bottom-start' },
            show: { type: Boolean },
			delay: { type: Number },
			tip: { type: Boolean },
			trigger: { type: String, values: ['hover', 'click'] },
			closeOnOutsideClick: { type: Boolean, attribute: 'close-outside-click' },
        }

        static styles = [
			super.styles,
			css`
				:host {
					--r-popup-z-index: 9999;
					--r-popup-arrow-color: var(--color-bg-overlay);
					--r-popup-arrow-border-color: var(--color-border-default);
					--r-popup-top-offset: var(--top-offset);
					
					display: contents;
				}

				.initial {
					all: initial;
				}
				
				.trigger-container {
					display: contents;
				}

				.popup {
					position: fixed;
					top: 0;
					left: 0;
					z-index: var(--r-popup-z-index);
					transition: transform ${POPUP_TRANSITION_MS}ms linear;
					transform-origin: top;
					display: flex;
				}
				
				.arrow {
					display: none;
					position: absolute;
					width: ${POPUP_OFFSET}px;
					height: ${POPUP_OFFSET}px;
					background: var(--r-popup-arrow-color);
				}
			`
		]
		
		render() {
			return html`
				<div id="trigger" class="trigger-container" @click="${this._onTriggerClick}" @mouseover="${this._onTriggerMouseOver}" @mouseout="${this._onTriggerMouseOut}">
					<slot @slotchange="${this._onTriggerSlotChange}"></slot>
				</div>
				
				<div style="position: absolute;">
					<div class="initial">
						<div id="popup-container" class="popup-container" style="display: none;" @mouseover="${this._onContainerMouseOver}" @mouseout="${this._onContainerMouseOut}">
							<div id="popup" class="popup">
								<slot name="content"></slot>
								<div id="arrow" class="arrow"></div>
							</div>
						</div>
					</div>
				</div>
			`;
		}
		
		get _triggerEl() {
			return this.shadowRoot.querySelector('#trigger');
		}
		get _popupEl() {
			return this.shadowRoot.querySelector('#popup');
		}
		get _popupContainerEl() {
			return this.shadowRoot.querySelector('#popup-container');
		}
		get _arrowEl() {
			return this.shadowRoot.querySelector('#arrow');
		}
		
		connectedCallback() {
			super.connectedCallback();
			this._onClick = (e) => {
				if (this.trigger === 'click' && this.closeOnOutsideClick) {
					if (!e.composedPath().includes(this)) {
						this.show = false;
					}
				}
			};
			window.addEventListener('click', this._onClick);
		}
		
		disconnectedCallback() {
			super.disconnectedCallback();
			window.removeEventListener('click', this._onClick);
			this._cleanupSlottedTriggerListeners();
		}
		
        updated(changed) {
			super.updated(changed);
			
            if (this.show) {
				this._open();
				this.dispatchEvent(new CustomEvent('open'));
            } else {
				this._close();
				this.dispatchEvent(new CustomEvent('close'));
            }
        }
		
		async _open() {
			if (!this.ready) {
				return;
			}
			
			if (!this.showing) {
				this.opening = true;
				this.showing = true;
				
				const { autoUpdate, computePosition, flip, arrow, offset, shift, size } = FloatingUIDOM;
				
				this._popupContainerEl.style.display = '';
				this._popupEl.style.transform = 'scale(0.9)';
				setTimeout(() => {
					this._popupEl.style.transform = 'scale(1)';
				});
				this._onOpenFinish = setTimeout(() => {
					this.opening = false;
				}, POPUP_TRANSITION_MS);
				
				let trigger = this._triggerEl.querySelector('slot').assignedElements()[0];
				if (trigger instanceof HTMLSlotElement) {
					trigger = trigger.assignedElements()[0];
				}
				if (!(trigger instanceof HTMLElement)) {
					throw new Error('Slotted trigger is not an element.');
				}
				
				const middleware = [
					offset(this.tip ? POPUP_OFFSET : 4),
					flip({
						padding: {
							top: parseFloat(getComputedStyle(this).getPropertyValue('--r-popup-top-offset') || 0),
						}
					}),
					shift(),
					
					// Ensure that popup content does not clip outside the viewport.
					// This also allows the popup content to be scrollable should its contents exceed the max dimensions.
					size({
						apply: ({ availableWidth, availableHeight, elements }) => {
							Object.assign(elements.floating.style, {
								maxWidth: `${availableWidth}px`,
								maxHeight: `${availableHeight}px`,
							});
						},
					}),
				];
				
				if (this.tip) {
					middleware.push(arrow({ element: this._arrowEl }));
				}
				
				const calculatePosition = placement => computePosition(trigger, this._popupEl, {
					middleware,
					placement,
					strategy: 'fixed',
				});
				
				this._cleanupPositioner = autoUpdate(trigger, this._popupEl, async () => {
					const { x, y, middlewareData, placement } = await calculatePosition(this.placement);
					Object.assign(this._popupEl.style, {
						left: `${x}px`,
						top: `${y}px`,
					});
					if (middlewareData.arrow) {
						const side = placement.split("-")[0];
						
						const staticSide = {
							top: 'bottom',
							right: 'left',
							bottom: 'top',
							left: 'right',
						}[side];
						
						const borderStyle = '1px solid var(--r-popup-arrow-border-color)';
						const border = {
							top: {
								border: 'none',
								borderBottom: borderStyle,
								borderRight: borderStyle,
							},
							bottom: {
								border: 'none',
								borderTop: borderStyle,
								borderLeft: borderStyle,
							},
							right: {
								border: 'none',
								borderLeft: borderStyle,
								borderBottom: borderStyle,
							},
							left: {
								border: 'none',
								borderRight: borderStyle,
								borderTop: borderStyle,
							}
						}[side];
						
						Object.assign(this._arrowEl.style, {
							display: 'block',
							left: middlewareData.arrow.x != null ? `${middlewareData.arrow.x}px` : '',
							top: middlewareData.arrow.y != null ? `${middlewareData.arrow.y}px` : '',
							[staticSide]: `${-this._arrowEl.offsetWidth / 2}px`,
							transform: 'rotate(45deg)',
							...border,
						});
					}
				});
			}
		}
		
		_close() {
			if (this.showing) {
				this._cleanupPositioner();
				
				if (this.opening) {
					this.opening = false;
					clearTimeout(this._onOpenFinish);
				}
				
				this._popupEl.style.transform = 'scale(0.9)';
				setTimeout(() => {
					this._popupContainerEl.style.display = 'none';
				}, POPUP_TRANSITION_MS);
				
				this.showing = false;
			}
		}
		
		toggle() {
			this.show = !this.show;
		}
		
		getSlottedTriggerElement() {
			let trigger = this._triggerEl.querySelector('slot').assignedElements()[0];
			if (trigger instanceof HTMLSlotElement) {
				trigger = trigger.assignedElements()[0];
			}
			return trigger;
		}
		
		_onContainerMouseOver() {
			this._mouseEnter();
		}
		_onContainerMouseOut() {
			this._mouseLeave();
		}
		
		_mouseEnter() {
			if (this.trigger === 'hover') {
				if (this.closeTimeout) {
					clearInterval(this.closeTimeout);
					this.closeTimeout = null;
				}
				if (this.delay) {
					setTimeout(() => {
						if (!this.closeTimeout) {
							this.show = true;
						}
					}, this.delay);
				} else {
					this.show = true;
				}
			}
			
		}
		_mouseLeave() {
			if (this.trigger === 'hover') {
				this.closeTimeout = setTimeout(() => {
					this.show = false;
				}, POPUP_CLOSE_DELAY_MS);
			}
		}
		
		_onTriggerMouseOver() {
			this._mouseEnter();
		}
		_onTriggerMouseOut() {
			this._mouseLeave();
		}
		
		_onTriggerClick() {
			if (this.trigger === 'click') {
				this.toggle();
			}
		}
		
		_onTriggerSlotChange() {
			this._cleanupSlottedTriggerListeners();
			
			this._slottedTriggerEl = this.getSlottedTriggerElement();
			
			if (this._slottedTriggerEl) {
				this._slottedTriggerEl.__rPopupFocusInEvent = () => this._mouseEnter();
				this._slottedTriggerEl.__rPopupFocusOutEvent = () => this._mouseLeave();
				
				this._slottedTriggerEl.addEventListener('focusin', this._slottedTriggerEl.__rPopupFocusInEvent);
				this._slottedTriggerEl.addEventListener('focusout', this._slottedTriggerEl.__rPopupFocusOutEvent);
			}
		}
		
		_cleanupSlottedTriggerListeners() {
			if (this._slottedTriggerEl) {
				this._slottedTriggerEl.removeEventListener('focusin', this._slottedTriggerEl.__rPopupFocusInEvent);
				this._slottedTriggerEl.removeEventListener('focusout', this._slottedTriggerEl.__rPopupFocusOutEvent);
				delete this._slottedTriggerEl.__rPopupFocusInEvent;
				delete this._slottedTriggerEl.__rPopupFocusOutEvent;
			}
		}
    }).define('r-popup');
})();
